#!/usr/bin/env python3
import os
import logging
import shutil
import re
from urllib.parse import quote

# MD_IMAGE_REGEX = r'!\[(?<altText>.*)\]\s*\((?<imageURL>.+)\)|img\s*src="(?<imageURL1>[^"]*)"\s*alt="(?<altText1>[^"]*)" \/>|img\s*alt="(?<altText2>[^"]*)"\s*src="(?<imageURL2>[^"]*)" \/>'
OBSIDIAN_IMAGE_REGEX = r'!\[\[((.+\.(png|jpe?g|avif|bmp|gif|svg|webp))(?:\|[0-9]+(?:x[0-9]+)?)?)\]\]'

# Initialise logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

# Get the base directory of the script
base_dir = os.path.dirname(os.path.realpath(__file__))
logger.info(f"Base path is {base_dir}")

# Define the source and target directories for posts
posts_source_dir = os.path.join(base_dir, "blog-vault", "posts")
posts_target_dir = os.path.join(base_dir, "content", "posts")
images_source_dir = os.path.join(base_dir, "blog-vault", "images")
images_target_dir = os.path.join(base_dir, "static", "images")

# Make sure all directories exist
for directory in [posts_source_dir, images_source_dir]:
    if not os.path.exists(directory):
        # Directory is important, so exit if not found
        logger.error(f"Directory {directory} does not exist. Cannot continue.")
        exit(1)

for directory in [posts_target_dir, images_target_dir]:
    if not os.path.exists(directory):
        # Directory is a target, just create it
        logger.warning(f"Directory {directory} does not exist, creating it")
        os.makedirs(directory)


# Copy all files from posts_source_dir to posts_target_dir
logger.info("Syncing blog posts...")
shutil.copytree(posts_source_dir, posts_target_dir, dirs_exist_ok=True)

# Fix images
logger.info("Fixing images...")
for filename in os.listdir(posts_target_dir):
    # Don't process non-markdown files
    if not filename.endswith(".md"):
        continue

    # Open file and read content
    filepath = os.path.join(posts_target_dir, filename)
    with open(filepath, "r") as file:
        content = file.read()
    
    # Find all images in the content
    images = re.findall(OBSIDIAN_IMAGE_REGEX, content)

    for image in images:
        # image[0] is the full match (e.g. filename.png|100x100)
        # image[1] is the image name (e.g. filename.png)
        # image[2] is the image extension (e.g. png)
        
        # Get the image name
        image_name = image[1]
        image_path = os.path.join(images_source_dir, image_name)

        logger.debug(f"Processing image {image_name}")
        logger.debug(f"Expected image path: {image_path}")

        # Check if the image exists
        if not os.path.exists(image_path):
            logger.warning(f"Image {image_name} not found")
            continue

        # Copy the image to the target directory
        target_image_path = os.path.join(images_target_dir, image_name)
        shutil.copy(image_path, target_image_path)

        # Replace the image path in the content
        image_url_name = quote(image_name)
        md_link = f"![{image_name}](/images/{image_url_name})"
        content = content.replace(f"![[{image[0]}]]", md_link)

    # Write the updated content back to the file
    with open(filepath, "w") as file:
        file.write(content)
    

logger.info("Done!")
