### IMPORTS
from ics import Calendar, Event, Attendee, ContentLine
import json
import logging
import datetime
import re


### INIT LOGGING
logging.basicConfig(format="%(asctime)s %(levelname)s %(message)s", datefmt="%Y-%m-%d %H:%M:%S")
logger = logging.getLogger("__name__")
logger.setLevel(logging.DEBUG)


### SUBPROGRAMS
def eval_lesson_condition(input_lesson: dict, condition: dict) -> bool:
    """
    Evaluates a condition on a lesson.
    """
    global logger
    return_value = False
    if condition["type"] == "and":
        # Evaluate and condition
        # Set default to true (we want to return true if all conditions are true)
        return_value = True
        # Loop through all conditions
        for i in condition["conditions"]:
            # If any condition is false, set return value to false and break
            if not eval_lesson_condition(input_lesson, i):
                return_value = False
                break
        
    elif condition["type"] == "or":
        # Evaluate or condition
        for i in condition["conditions"]:
            # If any condition is true, set return value to true and break
            if eval_lesson_condition(input_lesson, i):
                return_value = True
                break
        
    elif condition["type"] == "not":
        # Evaluate not condition
        return_value = not eval_lesson_condition(input_lesson, condition["condition"])

    elif condition["type"] == "field":
        # Evaluate a field condition
        fld_name = condition["field"]
        fld_expected_value = condition["value"]
        fld_operator = condition["operator"]

        # Only continue if the field exists
        if fld_name in input_lesson:
            fld_value = input_lesson[fld_name]
            # Evaluate the operator
            if fld_operator == "equals":
                return_value = fld_value == fld_expected_value
            
            elif fld_operator == "contains":
                return_value = fld_expected_value in fld_value
            
            elif fld_operator == "startswith":
                return_value = fld_value.startswith(fld_expected_value)
            
            elif fld_operator == "endswith":
                return_value = fld_value.endswith(fld_expected_value)
            
            elif fld_operator == ">":
                return_value = fld_value > fld_expected_value
            
            elif fld_operator == "<":
                return_value = fld_value < fld_expected_value
            
            elif fld_operator == ">=":
                return_value = fld_value >= fld_expected_value
            
            elif fld_operator == "<=":
                return_value = fld_value <= fld_expected_value
            
            else:
                # Invalid operator
                logger.warn("Invalid operator: %s, returning False.", fld_operator)
    else:
        # Invalid condition type
        logger.warn("Invalid condition type %s, returning False.", condition["type"])
    return return_value


def apply_patch(input_lessons: list, patch: dict, week_lookup_table: list) -> list:
    """
    Applies a patch to a list of lessons.
    """
    global logger
    # Process based on patch type
    if patch["type"] == "edit":
        # Edit all lessons that match the condition
        for lesson in input_lessons:
            if eval_lesson_condition(lesson, patch["condition"]):
                # Edit the lesson
                for key, value in patch["edit"].items():
                    lesson[key] = value
    elif patch["type"] == "edit_times":
        # Edit all lesson times that match the condition
        for lesson in input_lessons:
            if eval_lesson_condition(lesson, patch["condition"]):
                # Get startDate and endDate
                start_date = datetime.datetime.strptime(lesson["startDate"], "%Y-%m-%dT%H:%M:%S")
                end_date = datetime.datetime.strptime(lesson["endDate"], "%Y-%m-%dT%H:%M:%S")
                # Get start time and end time, replace in date
                if "start" in patch["edit"]:
                    start_time = datetime.time.fromisoformat(patch["edit"]["start"])
                    start_date = start_date.replace(hour=start_time.hour, minute=start_time.minute, second=start_time.second)
                if "end" in patch["edit"]:
                    end_time = datetime.time.fromisoformat(patch["edit"]["end"])
                    end_date = end_date.replace(hour=end_time.hour, minute=end_time.minute, second=end_time.second)
                # Set new dates
                lesson["startDate"] = start_date.strftime("%Y-%m-%dT%H:%M:%S")
                lesson["endDate"] = end_date.strftime("%Y-%m-%dT%H:%M:%S")
    elif patch["type"] == "add":
        # Add a lesson
        # Build day map
        week_lu_days = {}
        for day in week_lookup_table:
            week_lu_days[day["dayofDate"]] = day
        # Loop all days
        # TODO replace with looking up days in table2
        for day in patch["days"]:
            if day in week_lu_days:
                # Create new lesson
                new_lesson = {}

                # Set all fields in the new lesson
                for key, value in patch["data"].items():
                    new_lesson[key] = value
                
                # Set Week ID
                new_lesson["weekID"] = week_lu_days[day]["weekID"]

                # Set Week Beginning Date
                new_lesson["weekBeginningDate"] = week_lu_days[day]["weekBeginningDate"]

                # Set Day of Date
                new_lesson["dayOfDate"] = day

                # Get date values
                day_beginning_from = datetime.datetime.strptime(week_lu_days[day]["startDate"], "%Y-%m-%dT%H:%M:%S")
                start_time_parse = datetime.time.fromisoformat(patch["start_time"])
                length_mins = patch["length"]

                # Set beginning time to 00:00:00 in case it isn't set that way
                day_beginning_from.replace(hour=0, minute=0, second=0)

                # Calculate start date from day beginning and start time
                start_date = day_beginning_from + datetime.timedelta(hours=start_time_parse.hour, minutes=start_time_parse.minute)
                end_date = start_date + datetime.timedelta(minutes=length_mins)

                # Set start and end dates
                new_lesson["startDate"] = start_date.strftime("%Y-%m-%dT%H:%M:%S")
                new_lesson["endDate"] = end_date.strftime("%Y-%m-%dT%H:%M:%S")

                # Add lesson to list
                input_lessons.append(new_lesson)
    # Return modified lessons
    return input_lessons


### MAIN PROGRAM
if __name__ == "__main__":
    # Open config
    conf = json.load(open("config.json"))
    
    # Create calendar
    cal = Calendar()

    # Open JSON file
    input_json = json.load(open("input.json"))
    lessons_table = input_json["table"]
    week_lookup_table = input_json["table2"]

    # Open patches
    patches = json.load(open("patches.json"))

    # Loop through patches
    for patch in patches:
        lessons_table = apply_patch(lessons_table, patch, week_lookup_table)
    
    # Loop through lessons
    for lesson in lessons_table:
        # Prepare data
        e_name = f"[{lesson['periods']}] {lesson['subject']}"
        e_begin = datetime.datetime.strptime(lesson["startDate"], "%Y-%m-%dT%H:%M:%S")
        e_end = datetime.datetime.strptime(lesson["endDate"], "%Y-%m-%dT%H:%M:%S")
        e_location = lesson["room"]

        # Create attendees
        #school_attendee = Organizer(
        #    email=conf["school_name"]
        #)
        #school_attendee.user_type = "GROUP"

        class_attendee = Attendee(
            email=f"{lesson['class']}"
        )
        class_attendee.user_type = "GROUP"
        class_attendee.role = "REQ-PARTICIPANT"
        class_attendee.rsvp = "False" 
                        #     ^ Might be a temporary bug with ics, 
                        # setting to a string works so not gonna change it
        class_attendee.status = "ACCEPTED"

        teacher_attendee = Attendee(
            email=f"{lesson['teacherName']}"
        )
        teacher_attendee.user_type = "INDIVIDUAL"
        teacher_attendee.role = "REQ-PARTICIPANT"
        teacher_attendee.rsvp = "False"
        teacher_attendee.status = "ACCEPTED"

        e_attendees = [
            # school_attendee,
            class_attendee, 
            teacher_attendee
        ]

        # Create event
        event = Event(
            summary=e_name,
            begin=e_begin,
            end=e_end,
            location=e_location,
            attendees=e_attendees,
            created=datetime.datetime.now()
        )

        event.extra.append(ContentLine(name="COLOR", value=f"#{lesson['subjectColour']}"))

        # Add event to calendar
        cal.events.append(event)
    
    # Save calendar
    with open("output.ics", "w") as f:
        srl = cal.serialize()
        # Remove all empty lines
        filtered = filter(lambda x: not re.match(r'^\s*$', x), srl.split("\n"))
        f.write("".join(filtered))
    # Convert LF to CRLF (if necessary)
    data = open("output.ics", "rb").read()
    newdata = data.replace(b"\r", b"\n")
    newdata = newdata.replace(b"\n\n", b"\n")
    newdata = newdata.replace(b"\n", b"\r\n")
    f = open("output.ics", "wb")
    f.write(newdata)
    f.close()