import json
import os
import threading
import time
import queue
import base64

import pygame
from flask import request, Flask

import ledpatterns
from absled import ILedString
import screenwidgets
from config import PROP_AVAILABLE_WIDGETS, PROP_AVAILABLE_PATTERNS, PROP_WINDOW_SCALE, PROP_WINDOW_FULLSCREEN, \
    PROP_SCREEN_WIDTH, PROP_SCREEN_HEIGHT, PROP_HIDE_MOUSE, PROP_LEDS_DRIVER, PROP_LEDS_LENGTH, PROP_LEDS_LOCATION, \
    PROP_FLASK_DEBUG, PROP_FLASK_HOST, PROP_FLASK_PORT, hw_config
from utils.jsonencoding import get_cfg_from_request, decode_dict, encode_dict
from utils.net import json_response, verify_json_request
from utils.sequencing import validate_sequence
from utils.state import ScreenState

app = Flask(__name__)


module_config = {}
widget_sequence = []
pattern_sequence = []

def load_module_config():
    global module_config

    # Load defaults
    module_defaults = json.load(open("module_defaults.json"))
    module_config = {}
    for module_name, module_conf in module_defaults.items():
        add_conf = {}
        for setting, props in module_conf.items():
            add_conf[setting] = props.get("default", None)
        module_config[module_name] = add_conf

    # Load config over defaults
    if os.path.exists("module_config.json"):
        mconf = json.load(open("module_config.json"))
        for module_name, module_conf in mconf.items():
            if module_name in module_config:
                for setting, value in module_conf.items():
                    if setting in module_config[module_name]:
                        module_config[module_name][setting] = value

    # Save config
    with open("module_config.json", "w") as f:
        json.dump(module_config, f, indent=4)

def save_module_config():
    with open("module_config.json", "w") as f:
        json.dump(module_config, f, indent=4)

def load_sequences():
    global widget_sequence, pattern_sequence
    with open("sequence_default.json") as f:
        seq = json.load(f)
        widget_sequence = seq.get("widget", [])
        pattern_sequence = seq.get("pattern", [])

    if os.path.exists("sequence_config.json"):
        with open("sequence_config.json") as f:
            seq = json.load(f)
            widget_sequence = seq.get("widget", widget_sequence)
            pattern_sequence = seq.get("pattern", pattern_sequence)

def save_sequences():
    with open("sequence_config.json", "w") as f:
        json.dump({
            "widget": widget_sequence,
            "pattern": pattern_sequence
        }, f, indent=4)

mainthread_queue = queue.Queue()
display_queue = queue.Queue()
led_queue = queue.Queue()


##############
# Web Routes #
##############

@app.route("/")
def wr_index():
    return f"""
    <h1>LED Clock</h1>
    Available widgets: {PROP_AVAILABLE_WIDGETS}<br>
    Available patterns: {PROP_AVAILABLE_PATTERNS}<br>
    <a href="/api/get_config">Config</a>
    """

@app.route("/api/available_fonts")
def wr_available_fonts():
    return json_response(pygame.font.get_fonts())

@app.route("/api/available_widgets")
def wr_available_widgets():
    return json_response(PROP_AVAILABLE_WIDGETS)


@app.route("/api/get_state")
def wr_get_state():
    return json_response({}) # TODO

@app.route("/api/toggle_screen_state", methods=["POST"])
def wr_toggle_state():
    mainthread_queue.put("TOGGLE_SCREEN_STATE")
    return json_response({
        "status": "OK"
    })

@app.route("/api/available_patterns")
def wr_available_patterns():
    return json_response(PROP_AVAILABLE_PATTERNS)

@app.route("/api/set_widget", methods=["POST"])
def wr_set_widget():
    # Make sure the request is valid
    valid, validation_response = verify_json_request(request, ["widget"])
    if not valid:
        return validation_response

    # Verify pattern is valid and available
    widget_name = request.json.get("widget")
    if widget_name is None:
        return json_response({
            "error": "No widget provided"
        }, 400)

    if widget_name not in PROP_AVAILABLE_WIDGETS:
        return json_response({
            "error": f"Widget {widget_name} is not available"
        }, 400)

    # Get the config (or return error)
    config_b64, config_error = get_cfg_from_request(request)
    if config_error is not None:
        return config_error

    # Send the pattern to the LED thread
    display_queue.put("SET_WIDGET:" + widget_name + ":" + config_b64)

    # Return success
    return json_response({
        "status": "OK",
        "widget": widget_name
    })

@app.route("/api/set_pattern", methods=["POST"])
def wr_set_pattern():
    # Make sure the request is valid
    valid, validation_response = verify_json_request(request, ["pattern"])
    if not valid:
        return validation_response

    # Verify pattern is valid and available
    pattern_name = request.json.get("pattern")
    if pattern_name is None:
        return json_response({
            "error": "No pattern provided"
        }, 400)

    if pattern_name not in PROP_AVAILABLE_PATTERNS:
        return json_response({
            "error": f"Pattern {pattern_name} is not available"
        }, 400)

    # Get the config (or return error)
    config_b64, config_response = get_cfg_from_request(request)
    if config_response is not None:
        return config_response

    # Send the pattern to the LED thread
    led_queue.put("SET_PATTERN:" + pattern_name + ":" + config_b64)

    # Return success
    return json_response({
        "status": "OK",
        "pattern": pattern_name
    })

@app.route("/api/set_sequence", methods=["POST"])
def wr_set_sequence():
    # Make sure the request is valid
    valid, validation_response = verify_json_request(request, ["widget", "pattern"])
    if not valid:
        return validation_response

    # Make sure the widget and pattern sequences are valid
    widget_seq = request.json.get("widget")
    pattern_seq = request.json.get("pattern")
    if (
            (not isinstance(widget_seq, list)) or
            (not isinstance(pattern_seq, list)) or
            len(widget_seq) == 0 or
            len(pattern_seq) == 0
    ):
        return json_response({
            "error": "Invalid sequence provided"
        }, 400)

    # Make sure each widget is valid
    for w_index in range(len(widget_seq)):
        widget = widget_seq[w_index]
        if not validate_sequence(widget, PROP_AVAILABLE_WIDGETS):
            return json_response({
                "error": "Invalid widget in sequence, index " + str(w_index)
            }, 400)

    # Make sure each pattern is valid
    for p_index in range(len(pattern_seq)):
        pattern = pattern_seq[p_index]
        if not validate_sequence(pattern, PROP_AVAILABLE_PATTERNS):
            return json_response({
                "error": "Invalid pattern in sequence, index " + str(p_index)
            }, 400)

    # Set the sequences
    mainthread_queue.put("SAVE_SEQUENCE_CFG:" + encode_dict(widget_seq) + ":" + encode_dict(pattern_seq))
    return json_response({
        "status": "OK"
    })

@app.route("/api/get_config")
def wr_get_config():
    return json_response(module_config)


#####################
# Loading functions #
#####################

def load_widget(widget_name: str, config_override: dict=None):
    # Get the widget properties from the config file
    widget_props = module_config.get(widget_name, {}).copy()

    if config_override is not None:
        widget_props.update(config_override)

    # Scale the properties that are supposed to be scalable
    for k, v in widget_props.copy().items():
        if (isinstance(v, int) or isinstance(v, float)) and k.startswith("scalable_"):
            widget_props[k.removeprefix("scalable_")] = v * PROP_WINDOW_SCALE

    # Add the pygame module to the widget properties
    widget_props["pygame"] = pygame

    # Load the widget
    return getattr(screenwidgets, widget_name)(widget_props)

def load_pattern(pattern_name: str, leds: ILedString, config_override: dict=None):
    pattern_props = module_config.get(pattern_name, {}).copy()

    if config_override is not None:
        pattern_props.update(config_override)

    return getattr(ledpatterns, pattern_name)(pattern_props, leds)


##################
# Loop Functions #
##################

def run_screen(start_widget: dict):
    # Initialise Pygame and create a window
    pygame.init()

    pg_options = 0x0
    if PROP_WINDOW_FULLSCREEN:
        pg_options |= pygame.FULLSCREEN
    screen = pygame.display.set_mode(
        (PROP_SCREEN_WIDTH * PROP_WINDOW_SCALE, PROP_SCREEN_HEIGHT * PROP_WINDOW_SCALE),
        pg_options
    )
    pygame.display.set_caption("Clock")
    pygame.mouse.set_visible(not PROP_HIDE_MOUSE)

    # Create a clock for FPS counter and timing
    clock = pygame.time.Clock()

    # Init widgets
    widget_to_load = start_widget.get("type")
    config_override = start_widget.get("config", {})

    current_widget = load_widget(widget_to_load, config_override)

    # Main loop
    running = True
    while running:
        for event in pygame.event.get():
            if event.type == pygame.QUIT:
                running = False

        # Clear the screen
        screen.fill((0, 0, 0))

        # Check for messages
        try:
            message = display_queue.get_nowait()
            parts = message.split(":")
            if parts[0] == "SET_WIDGET":
                widget_to_load = parts[1]
                if widget_to_load in PROP_AVAILABLE_WIDGETS: # Check input validity
                    config_override = {}
                    if len(parts) == 3:
                        try:
                            config_override = json.loads(base64.b64decode(parts[2]).decode("utf-8"))
                        except json.JSONDecodeError:
                            pass
                    current_widget = load_widget(widget_to_load, config_override)
        except queue.Empty:
            pass

        # Draw stuff here
        current_widget.draw(screen)

        # Display FPS counter
        if hw_config["show_fps"]:
            fps = clock.get_fps()
            font = pygame.font.Font(None, 36)
            text = font.render(f"FPS: {fps:.2f}", True, (255, 255, 255))
            screen.blit(text, (10, 10))

        # Tick the clock
        if hw_config["fps_max"] > 0:
            clock.tick(hw_config["fps_max"])
        else:
            clock.tick()

        # Update the screen
        pygame.display.flip()

    # Send exit message to main thread
    mainthread_queue.put("EXIT")

    # Quit Pygame
    pygame.quit()

def run_leds(start_pattern: dict):
    # Load LEDs
    if PROP_LEDS_DRIVER == "dummy":
        from absled.dummy import DummyLedString
        leds = DummyLedString("", PROP_LEDS_LENGTH)
    elif PROP_LEDS_DRIVER == "npixel":
        from absled.npixel import NeopixelLedString
        leds = NeopixelLedString(PROP_LEDS_LOCATION, PROP_LEDS_LENGTH)
    else:
        raise ValueError("Invalid LED driver")

    pattern_name = start_pattern.get("type")
    config_override = start_pattern.get("config", {})

    current_pattern = load_pattern(pattern_name, leds, config_override)
    pattern_timer = module_config.get(pattern_name, {}).get("tick_rate", -1)
    current_pattern.tick()

    while True:
        try:
            message = led_queue.get_nowait()
            parts = message.split(":")
            if parts[0] == "SET_PATTERN":
                pattern_name = parts[1]
                if pattern_name in PROP_AVAILABLE_PATTERNS:
                    config_override = {}
                    if len(parts) == 3:
                        try:
                            config_override = json.loads(base64.b64decode(parts[2]).decode("utf-8"))
                        except json.JSONDecodeError:
                            pass
                    current_pattern = load_pattern(pattern_name, leds, config_override)
                    pattern_timer = module_config.get(pattern_name, {}).get("tick_rate", -1)
                    current_pattern.tick()
            elif message == "EXIT":
                break
        except queue.Empty:
            pass

        if pattern_timer >= 0:
            time.sleep(pattern_timer / 1000)
            current_pattern.tick()


if __name__ == "__main__":
    # Load configuration
    load_module_config()
    load_sequences()

    if len(widget_sequence) == 0:
        raise ValueError("No widgets in sequence")
    if len(pattern_sequence) == 0:
        raise ValueError("No patterns in sequence")

    screen_state = ScreenState.WIDGET_CYCLE

    initial_widget = widget_sequence[0]
    initial_pattern = pattern_sequence[0]

    current_widget_index = 0
    current_pattern_index = 0

    current_widget_delta = 0
    current_pattern_delta = 0

    force_change_widget = False
    force_change_pattern = False

    # Fork off pygame
    pygame_thread = threading.Thread(
        target=lambda: run_screen(initial_widget),
        daemon=True
    )
    pygame_thread.start()

    # Fork off flask server
    flask_thread = threading.Thread(
        target=lambda: app.run(debug=PROP_FLASK_DEBUG, host=PROP_FLASK_HOST, port=PROP_FLASK_PORT, use_reloader=False),
        daemon=True
    )
    flask_thread.start()

    # Fork off LED thread
    led_thread = threading.Thread(
        target=lambda: run_leds(initial_pattern),
        daemon=True
    )
    led_thread.start()

    # Wait for the threads to finish
    try:
        while True:
            # Process message queue
            try:
                message = mainthread_queue.get_nowait()
                parts = message.split(":")
                print("Main Message Received")
                print(message)
                if message == "EXIT":
                    display_queue.put("EXIT")
                    led_queue.put("EXIT")
                    break
                elif parts[0] == "UPDATE_ALL_MODULE_CFG" and len(parts) == 2:
                    module_config = decode_dict(parts[1])
                    save_module_config()
                elif parts[0] == "SAVE_SEQUENCE_CFG" and len(parts) == 3:
                    widget_sequence = decode_dict(parts[1])
                    pattern_sequence = decode_dict(parts[2])
                    if len(widget_sequence) <= current_widget_index:
                        current_widget_index = 0
                    if len(pattern_sequence) <= current_pattern_index:
                        current_pattern_index = 0
                    save_sequences()
                elif message == "CYCLE_WIDGET":
                    force_change_widget = True
                elif message == "CYCLE_PATTERN":
                    force_change_pattern = True
                elif message == "TOGGLE_SCREEN_STATE":
                    if screen_state == ScreenState.WIDGET_CYCLE:
                        screen_state = ScreenState.WIDGET_SINGLE
                    elif screen_state == ScreenState.WIDGET_SINGLE:
                        screen_state = ScreenState.WIDGET_CYCLE
                    else:
                        pass

            except queue.Empty:
                pass
            # Check if time delta has passed
            if screen_state == ScreenState.WIDGET_CYCLE:
                # For Widgets
                curr_widget = widget_sequence[current_widget_index]
                if curr_widget.get("duration", -1) != -1 or force_change_widget:
                    # Check if the widget has existed for too long
                    if curr_widget.get("duration") * 1000 <= current_widget_delta or force_change_widget:
                        # Set the widget to the next widget
                        current_widget_index += 1
                        if current_widget_index >= len(widget_sequence):
                            current_widget_index = 0
                        display_queue.put("SET_WIDGET:" +
                                          widget_sequence[current_widget_index].get("type") + ":" +
                                          encode_dict(widget_sequence[current_widget_index].get("config_override", {}))
                                          )
                        current_widget_delta = 0
                        force_change_widget = False
                # For patterns
                curr_pattern = pattern_sequence[current_pattern_index]
                if curr_pattern.get("duration", -1) != -1 or force_change_pattern:
                    # Check if the pattern has existed for too long
                    if curr_pattern.get("duration") * 1000 <= current_pattern_delta or force_change_pattern:
                        # Set the pattern to the next pattern
                        current_pattern_index += 1
                        if current_pattern_index >= len(pattern_sequence):
                            current_pattern_index = 0
                        led_queue.put("SET_PATTERN:" +
                                      pattern_sequence[current_pattern_index].get("type") + ":" +
                                      encode_dict(pattern_sequence[current_pattern_index].get("config_override", {}))
                                      )
                        current_pattern_delta = 0
                        force_change_pattern = False

            # Keep main thread alive
            current_widget_delta += hw_config["main_update_frequency"]
            time.sleep(hw_config["main_update_frequency"] / 1000)
    except KeyboardInterrupt:
        exit()
    exit()
