import json
import socket
from typing import Any

from flask import Response, Request


def get_ip() -> str:
    """
    Gets the IP address of the current machine
    :return: the IP address as a string
    """
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    s.settimeout(0)
    try:
        # doesn't even have to be reachable
        s.connect(('10.254.254.254', 1))
        ip_addr = s.getsockname()[0]
    except Exception:
        ip_addr = '127.0.0.1'
    finally:
        s.close()
    return ip_addr


def json_response(data: Any, status=200) -> Response:
    """
    Generate a JSON response from any json-serializable data
    :param data: The data to be sent
    :param status: The status code of the response
    :return: A JSON response
    """
    return Response(json.dumps(data), status=status, mimetype="application/json")


def verify_json_request(req: Request, required_fields: list|None = None) -> tuple[bool, Response|None]:
    """
    Verify that a request is JSON and contains all required fields
    :param req: The request to verify
    :param required_fields: A list of required fields, or None
    :return: A tuple containing a boolean indicating if the request is valid and a response if it is not
    """
    if required_fields is None:
        required_fields = []

    if not req.is_json:
        return False, json_response({
            "error": "Request is not JSON. Please make sure mimetype is set to application/json."
        }, 400)

    for field in required_fields:
        if field not in req.json:
            return False, json_response({
                "error": f"Missing field {field}"
            }, 400)

    return True, None